<?php
/**
 * Plugin Name: DrGlenn Code
 * Description: Single-file utility to add sitewide CSS & JS, inject safe header/footer HTML, and provide reusable HTML/CSS/JS snippets with shortcodes. Optional PHP snippets are admin-rendered and cached for visitors, with strong safety guards.
 * Version: 2.3
 * Author: DrGlenn
 * License: GPL-2.0+
 * Requires at least: 5.5
 * Requires PHP: 7.2
 * Text Domain: https://fixmyhackedwebsite.com
 * Update URI: false
 */

if ( ! defined( 'ABSPATH' ) ) exit;

// -----------------------------------------------------------------------------
// CONSTANTS & DEFAULTS
// -----------------------------------------------------------------------------
const DGC_VER = '2.3';
const DGC_OPT = 'dgc_options_v1';

function dgc_defaults(){
	return array(
		'disable_all' => 0,
		'php_enabled' => 0, // master toggle for PHP snippets (off by default)
		'php_safe_off' => 0, // allow turning off PHP snippets quickly
		'header_html' => '',
		'footer_html' => '',
		'site_css'    => '',
		'site_js'     => '',
		'snip_html'   => array(), // [ id => html ]
		'snip_css'    => array(), // [ id => css ]
		'snip_js'     => array(), // [ id => js ]
		'snip_php'    => array(), // [ id => php_code ]
	);
}

register_activation_hook( __FILE__, function(){
	$opt = get_option( DGC_OPT );
	if ( ! is_array( $opt ) ) update_option( DGC_OPT, dgc_defaults() );
});

function dgc_opts(){
	$def = dgc_defaults();
	$opt = get_option( DGC_OPT, array() );
	return array_merge( $def, is_array($opt) ? $opt : array() );
}

// -----------------------------------------------------------------------------
// CAPABILITIES & SANITIZE
// -----------------------------------------------------------------------------
function dgc_can_manage(){
	return current_user_can( 'manage_options' );
}
function dgc_can_unfiltered(){
	return current_user_can( 'unfiltered_html' );
}

add_action( 'admin_init', function(){
	register_setting( 'dgc_group', DGC_OPT, array( 'sanitize_callback' => 'dgc_sanitize' ) );
});

function dgc_sanitize( $in ){
	$old = dgc_opts();
	$out = dgc_defaults();
	if ( ! dgc_can_manage() ) return $old;

	$out['disable_all'] = empty( $in['disable_all'] ) ? 0 : 1;
	$out['php_enabled'] = empty( $in['php_enabled'] ) ? 0 : 1;
	$out['php_safe_off'] = empty( $in['php_safe_off'] ) ? 0 : 1;

	// Only allow raw HTML/JS/CSS edits to users with unfiltered_html
	if ( dgc_can_unfiltered() ) {
		$out['header_html'] = isset($in['header_html']) ? (string) $in['header_html'] : '';
		$out['footer_html'] = isset($in['footer_html']) ? (string) $in['footer_html'] : '';
		$out['site_css']    = isset($in['site_css'])    ? (string) $in['site_css']    : '';
		$out['site_js']     = isset($in['site_js'])     ? (string) $in['site_js']     : '';
	} else {
		// Keep previous values if user lacks capability
		$out['header_html'] = $old['header_html'];
		$out['footer_html'] = $old['footer_html'];
		$out['site_css']    = $old['site_css'];
		$out['site_js']     = $old['site_js'];
	}

	// Snippets (HTML/CSS/JS): arrays of id => content
	$out['snip_html'] = is_array( $old['snip_html'] ) ? $old['snip_html'] : array();
	$out['snip_css']  = is_array( $old['snip_css'] )  ? $old['snip_css']  : array();
	$out['snip_js']   = is_array( $old['snip_js'] )   ? $old['snip_js']   : array();
	$out['snip_php']  = is_array( $old['snip_php'] )  ? $old['snip_php']  : array();

	// Handle add/update/delete operations (submitted via simple forms)
	$ops = isset( $in['ops'] ) && is_array( $in['ops'] ) ? $in['ops'] : array();
	if ( dgc_can_unfiltered() ) {
		foreach ( array('html','css','js') as $type ) {
			$add_id = isset($ops["add_{$type}_id"]) ? sanitize_key($ops["add_{$type}_id"]) : '';
			$add_val = isset($ops["add_{$type}_val"]) ? (string) $ops["add_{$type}_val"] : '';
			if ( $add_id !== '' && $add_val !== '' ) {
				$out["snip_{$type}"][$add_id] = $add_val;
			}
			// updates
			if ( isset($ops["list_{$type}"]) && is_array($ops["list_{$type}"]) ) {
				foreach ( $ops["list_{$type}"] as $id => $val ) {
					$id = sanitize_key($id);
					$out["snip_{$type}"][$id] = (string) $val;
				}
			}
			// deletes
			if ( isset($ops["del_{$type}"]) && is_array($ops["del_{$type}"]) ) {
				foreach ( $ops["del_{$type}"] as $id => $on ) {
					$id = sanitize_key($id);
					unset( $out["snip_{$type}"][$id] );
				}
			}
		}
	}

	// PHP snippets are stored but only executed via guarded path
	if ( dgc_can_unfiltered() ) {
		$add_pid = isset($ops['add_php_id']) ? sanitize_key($ops['add_php_id']) : '';
		$add_pval = isset($ops['add_php_val']) ? (string) $ops['add_php_val'] : '';
		if ( $add_pid !== '' && $add_pval !== '' ) {
			$out['snip_php'][$add_pid] = $add_pval;
		}
		if ( isset($ops['list_php']) && is_array($ops['list_php']) ) {
			foreach ( $ops['list_php'] as $id => $val ) {
				$id = sanitize_key($id);
				$out['snip_php'][$id] = (string) $val;
			}
		}
		if ( isset($ops['del_php']) && is_array($ops['del_php']) ) {
			foreach ( $ops['del_php'] as $id => $on ) {
				$id = sanitize_key($id);
				unset( $out['snip_php'][$id] );
			}
		}
	}

	return $out;
}

// -----------------------------------------------------------------------------
// ADMIN PAGE
// -----------------------------------------------------------------------------
add_action( 'admin_menu', function(){
	add_options_page(
		'\x44rGlenn Code',
		'\x44rGlenn Code',
		'manage_options',
		'drglenn-code',
		'dgc_render_settings_page'
	);
});

function dgc_render_settings_page(){
	if ( ! dgc_can_manage() ) wp_die( esc_html__( 'You do not have permission.', 'drglenn-code' ) );
	$opt = dgc_opts();
	?>
	<div class="wrap">
		<h1>DrGlenn Code <small style="font-size:12px;opacity:.7;">v<?php echo esc_html( DGC_VER ); ?></small></h1>
		<p style="max-width:820px;color:#556480;">Lightweight, single-file helper for safe, sitewide customization. Add CSS & JS, inject header/footer HTML, and use small reusable snippets. <strong>Optional PHP</strong> snippets run only for admins and are cached for visitors with strict safety guards.</p>

		<?php if ( isset($_GET['settings-updated']) ) : ?>
			<div class="notice notice-success"><p>Settings saved.</p></div>
		<?php endif; ?>

		<form method="post" action="options.php">
			<?php settings_fields( 'dgc_group' ); ?>
			<?php $name = DGC_OPT; ?>

			<h2>Global Output</h2>
			<table class="form-table" role="presentation">
				<tr>
					<th scope="row">Disable All Output</th>
					<td><label><input type="checkbox" name="<?php echo esc_attr($name); ?>[disable_all]" value="1" <?php checked(1, $opt['disable_all']); ?>> Panic switch</label></td>
				</tr>
				<tr>
					<th scope="row"><label for="dgc_header_html">Header HTML</label></th>
					<td>
						<textarea id="dgc_header_html" name="<?php echo esc_attr($name); ?>[header_html]" class="large-text code" rows="6" placeholder="&lt;meta name=&quot;google-site-verification&quot; content=&quot;...&quot;&gt;\n&lt;script&gt;...&lt;/script&gt;"><?php echo esc_textarea($opt['header_html']); ?></textarea>
						<p class="description">Printed inside <code>&lt;head&gt;</code> via <code>wp_head</code>. <strong>Admins with unfiltered_html only.</strong></p>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="dgc_site_css">Sitewide CSS</label></th>
					<td>
						<textarea id="dgc_site_css" name="<?php echo esc_attr($name); ?>[site_css]" class="large-text code" rows="6" placeholder=".selector{display:none}"><?php echo esc_textarea($opt['site_css']); ?></textarea>
						<p class="description">Injected into <code>&lt;style id="dgc-site-css"&gt;</code> in the head.</p>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="dgc_footer_html">Footer HTML</label></th>
					<td>
						<textarea id="dgc_footer_html" name="<?php echo esc_attr($name); ?>[footer_html]" class="large-text code" rows="6" placeholder="&lt;script&gt;...&lt;/script&gt;"><?php echo esc_textarea($opt['footer_html']); ?></textarea>
						<p class="description">Printed before <code>&lt;/body&gt;</code> via <code>wp_footer</code>. <strong>Admins with unfiltered_html only.</strong></p>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="dgc_site_js">Sitewide JS</label></th>
					<td>
						<textarea id="dgc_site_js" name="<?php echo esc_attr($name); ?>[site_js]" class="large-text code" rows="6" placeholder="document.addEventListener('DOMContentLoaded',()=>{ /* ... */ })"><?php echo esc_textarea($opt['site_js']); ?></textarea>
						<p class="description">Injected into <code>&lt;script id="dgc-site-js"&gt;</code> before <code>&lt;/body&gt;</code>.</p>
					</td>
				</tr>
			</table>

			<h2>Reusable Snippets</h2>
			<p class="description">Use these with shortcodes: <code>[dgc_html id="hero-note"]</code>, <code>[dgc_css id="card-tweak"]</code>, <code>[dgc_js id="gallery-init"]</code>. CSS/JS snippets are output once per page (in head/footer), even if shortcode appears multiple times.</p>

			<?php foreach ( array('html' => 'HTML','css' => 'CSS','js' => 'JS') as $tkey => $tlabel ) : ?>
				<h3><?php echo esc_html( $tlabel ); ?> Snippets</h3>
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row">Add New</th>
						<td>
							<input type="text" name="<?php echo esc_attr($name); ?>[ops][add_<?php echo esc_attr($tkey); ?>_id]" placeholder="id" style="width:160px;">
							<br>
							<textarea name="<?php echo esc_attr($name); ?>[ops][add_<?php echo esc_attr($tkey); ?>_val]" class="large-text code" rows="4" placeholder="Snippet content..."></textarea>
						</td>
					</tr>
					<?php $arr = isset($opt['snip_'.$tkey]) && is_array($opt['snip_'.$tkey]) ? $opt['snip_'.$tkey] : array(); ?>
					<?php foreach ( $arr as $id => $val ) : $id_key = sanitize_key($id); ?>
					<tr>
						<th scope="row"><label><?php echo esc_html($id); ?></label><br><label><input type="checkbox" name="<?php echo esc_attr($name); ?>[ops][del_<?php echo esc_attr($tkey); ?>][<?php echo esc_attr($id_key); ?>]" value="1"> Delete</label></th>
						<td>
							<textarea name="<?php echo esc_attr($name); ?>[ops][list_<?php echo esc_attr($tkey); ?>][<?php echo esc_attr($id_key); ?>]" class="large-text code" rows="4"><?php echo esc_textarea($val); ?></textarea>
						</td>
					</tr>
					<?php endforeach; ?>
				</table>
			<?php endforeach; ?>

			<h2>Optional PHP Snippets (Advanced)</h2>
			<p class="description"><strong>Off by default.</strong> For safety: snippets run only for admins; visitors see the <em>cached output</em>. On the first render or if the cache is missing, public output is empty. Dangerous functions are blocked; any error disables PHP snippets until reviewed.</p>
			<table class="form-table" role="presentation">
				<tr>
					<th scope="row">Enable PHP snippets</th>
					<td><label><input type="checkbox" name="<?php echo esc_attr($name); ?>[php_enabled]" value="1" <?php checked(1,$opt['php_enabled']); ?>> I understand the risks</label> &nbsp; &nbsp; <label><input type="checkbox" name="<?php echo esc_attr($name); ?>[php_safe_off]" value="1" <?php checked(1,$opt['php_safe_off']); ?>> Disable PHP snippets (panic)</label></td>
				</tr>
			</table>

			<h3>PHP Snippets</h3>
			<table class="form-table" role="presentation">
				<tr>
					<th scope="row">Add New</th>
					<td>
						<input type="text" name="<?php echo esc_attr($name); ?>[ops][add_php_id]" placeholder="id" style="width:160px;">
						<br>
						<textarea name="<?php echo esc_attr($name); ?>[ops][add_php_val]" class="large-text code" rows="4" placeholder="return 'Hello World';"></textarea>
						<p class="description">Must <code>return</code> a string. Example: <code>return date('Y');</code></p>
					</td>
				</tr>
				<?php $arrp = isset($opt['snip_php']) && is_array($opt['snip_php']) ? $opt['snip_php'] : array(); ?>
				<?php foreach ( $arrp as $id => $val ) : $id_key = sanitize_key($id); ?>
				<tr>
					<th scope="row"><label><?php echo esc_html($id); ?></label><br><label><input type="checkbox" name="<?php echo esc_attr($name); ?>[ops][del_php][<?php echo esc_attr($id_key); ?>]" value="1"> Delete</label></th>
					<td>
						<textarea name="<?php echo esc_attr($name); ?>[ops][list_php][<?php echo esc_attr($id_key); ?>]" class="large-text code" rows="4"><?php echo esc_textarea($val); ?></textarea>
						<p class="description">Shortcode: <code>[dgc_php id="<?php echo esc_html($id); ?>"]</code> — runs for admins, cached for visitors.</p>
					</td>
				</tr>
				<?php endforeach; ?>
			</table>

			<?php submit_button(); ?>
		</form>

		<hr>
		<h2>Shortcodes</h2>
		<ul>
			<li><code>[dgc_html id="note"]</code> → prints stored HTML snippet (sanitized to allowed tags).</li>
			<li><code>[dgc_css id="card-tweak"]</code> → queues a CSS snippet to be printed once in the head.</li>
			<li><code>[dgc_js id="gallery"]</code> → queues a JS snippet to be printed once in the footer.</li>
			<li><code>[dgc_php id="calc"]</code> → outputs cached string. Admins populate cache by visiting the page.</li>
		</ul>
	</div>
	<?php
}

// -----------------------------------------------------------------------------
// FRONTEND OUTPUT
// -----------------------------------------------------------------------------
function dgc_is_disabled(){
	$opt = dgc_opts();
	if ( ! empty($_GET['dgc_safe']) ) return true; // request-safe mode
	return !empty( $opt['disable_all'] );
}

add_action( 'wp_head', function(){
	if ( dgc_is_disabled() ) return;
	$opt = dgc_opts();
	static $done = false; if ( $done ) return; $done = true;
	// Header HTML (raw) — only if set
	if ( ! empty( $opt['header_html'] ) ) {
		echo "\n<!-- DrGlenn Code header -->\n";
		echo $opt['header_html'] . "\n"; // trusted for admins who saved it
	}
	// Site CSS
	if ( ! empty( $opt['site_css'] ) || ! empty( $GLOBALS['dgc_css_queue'] ) ) {
		echo "\n<style id=\"dgc-site-css\">\n";
		if ( ! empty( $opt['site_css'] ) ) echo $opt['site_css'] . "\n";
		if ( ! empty( $GLOBALS['dgc_css_queue'] ) && is_array($GLOBALS['dgc_css_queue']) ) {
			$css = dgc_opts()['snip_css'];
			foreach ( array_unique($GLOBALS['dgc_css_queue']) as $sid ) {
				$sid = sanitize_key($sid);
				if ( isset($css[$sid]) && $css[$sid] !== '' ) echo $css[$sid] . "\n";
			}
		}
		echo "</style>\n";
	}
}, 99 );

add_action( 'wp_footer', function(){
	if ( dgc_is_disabled() ) return;
	$opt = dgc_opts();
	static $done = false; if ( $done ) return; $done = true;
	// Footer HTML (raw)
	if ( ! empty( $opt['footer_html'] ) ) {
		echo "\n<!-- DrGlenn Code footer -->\n";
		echo $opt['footer_html'] . "\n"; // trusted for admins who saved it
	}
	// Site JS + queued JS snippets
	if ( ! empty( $opt['site_js'] ) || ! empty( $GLOBALS['dgc_js_queue'] ) ) {
		echo "\n<script id=\"dgc-site-js\">\n";
		if ( ! empty( $opt['site_js'] ) ) echo $opt['site_js'] . "\n";
		if ( ! empty( $GLOBALS['dgc_js_queue'] ) && is_array($GLOBALS['dgc_js_queue']) ) {
			$js = dgc_opts()['snip_js'];
			foreach ( array_unique($GLOBALS['dgc_js_queue']) as $sid ) {
				$sid = sanitize_key($sid);
				if ( isset($js[$sid]) && $js[$sid] !== '' ) echo $js[$sid] . "\n";
			}
		}
		echo "</script>\n";
	}
}, 99 );

// -----------------------------------------------------------------------------
// SHORTCODES (HTML/CSS/JS/PHP)
// -----------------------------------------------------------------------------
add_shortcode( 'dgc_html', function( $atts ){
	if ( dgc_is_disabled() ) return '';
	$atts = shortcode_atts( array( 'id' => '' ), $atts, 'dgc_html' );
	$id = sanitize_key( $atts['id'] );
	if ( ! $id ) return '';
	$opt = dgc_opts(); $arr = isset($opt['snip_html']) ? $opt['snip_html'] : array();
	if ( empty( $arr[$id] ) ) return '';
	// Allow a safe subset of HTML in content
	return wp_kses_post( $arr[$id] );
});

add_shortcode( 'dgc_css', function( $atts ){
	if ( dgc_is_disabled() ) return '';
	$atts = shortcode_atts( array( 'id' => '' ), $atts, 'dgc_css' );
	$id = sanitize_key( $atts['id'] );
	if ( ! $id ) return '';
	$GLOBALS['dgc_css_queue'][] = $id;
	return '';
});

add_shortcode( 'dgc_js', function( $atts ){
	if ( dgc_is_disabled() ) return '';
	$atts = shortcode_atts( array( 'id' => '' ), $atts, 'dgc_js' );
	$id = sanitize_key( $atts['id'] );
	if ( ! $id ) return '';
	$GLOBALS['dgc_js_queue'][] = $id;
	return '';
});

add_shortcode( 'dgc_php', function( $atts ){
	if ( dgc_is_disabled() ) return '';
	$atts = shortcode_atts( array( 'id' => '' ), $atts, 'dgc_php' );
	$id = sanitize_key( $atts['id'] );
	if ( ! $id ) return '';
	$opt = dgc_opts();
	if ( empty( $opt['php_enabled'] ) || ! empty( $opt['php_safe_off'] ) ) return '';
	// Admin executes & updates cache; visitors read from cache only
	$code = isset($opt['snip_php'][$id]) ? (string) $opt['snip_php'][$id] : '';
	if ( $code === '' ) return '';
	$hash = md5( $code );
	$transient_key = 'dgc_php_cache_' . $id . '_' . $hash;
	$cached = get_transient( $transient_key );
	if ( false !== $cached && is_string( $cached ) ) return $cached;
	if ( ! current_user_can( 'manage_options' ) ) return '';
	// Admin: attempt to render, store in cache, return
	$out = dgc_eval_php_snippet( $code, $err );
	if ( $err ) {
		update_option( 'dgc_php_last_error', current_time('mysql') . ' — ' . $err );
		// Disable PHP snippets to prevent loops
		$opt['php_safe_off'] = 1; update_option( DGC_OPT, $opt );
		return '';
	}
	set_transient( $transient_key, $out, HOUR_IN_SECONDS * 24 );
	return $out;
});

// Guarded evaluator for PHP snippets
function dgc_eval_php_snippet( $code, &$error = null ){
	$error = null;
	// quick denylist (best-effort)
	$deny = '/(\\b(?:eval|assert|exec|system|shell_exec|passthru|proc_open|popen|pcntl_\\w+|curl_exec|curl_multi_exec|parse_ini_\\w+|file_put_contents|fopen|fwrite|unlink|rmdir|rename|chmod|chown|copy|move_uploaded_file)\\b|`|base64_decode\s*\(|gz(un)?compress\s*\()/i';
	if ( preg_match( $deny, $code ) ) {
		$error = 'Blocked dangerous token in PHP snippet.';
		return '';
	}
	$code = trim( preg_replace('/^<\?php|\?>$/i','',$code) );
	try {
		ob_start();
		// Wrap inside a function scope and ensure a string is returned
		$result = eval('return (function(){ ' . $code . ' })();');
		$buffer = ob_get_clean();
		return (string) $buffer . (string) $result;
	} catch ( Throwable $t ) {
		ob_end_clean();
		$error = $t->getMessage();
		return '';
	}
}

// Admin notice for last PHP error
add_action('admin_notices', function(){
	if ( ! current_user_can('manage_options') ) return;
	$err = get_option('dgc_php_last_error');
	if ( $err ) {
		echo '<div class="notice notice-error"><p><strong>DrGlenn Code:</strong> PHP snippets disabled due to error: ' . esc_html($err) . '</p></div>';
		delete_option('dgc_php_last_error');
	}
});

// -----------------------------------------------------------------------------
// MISC
// -----------------------------------------------------------------------------
// Clear CSS/JS queues each request (explicit)
$GLOBALS['dgc_css_queue'] = array();
$GLOBALS['dgc_js_queue']  = array();

// Helpful links on plugins list
add_filter('plugin_action_links_' . plugin_basename(__FILE__), function($links){
	$links[] = '<a href="' . esc_url( admin_url('options-general.php?page=drglenn-code') ) . '">Settings</a>';
	$links[] = '<a href="https://fixmyhackedwebsite.com/drg-code/drg-code.zip" target="_blank" rel="noopener">Download ZIP</a>';
	return $links;
});
